"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.requireSuperAdmin = exports.requireAdmin = exports.requireVendor = exports.requireRole = exports.authenticateUser = void 0;
const supabase_1 = require("@/config/supabase");
/**
 * Middleware to authenticate user via Supabase token
 * Extracts token from Authorization header and verifies with Supabase
 */
const authenticateUser = async (req, res, next) => {
    try {
        // Get token from Authorization header
        const authHeader = req.headers.authorization;
        if (!authHeader || !authHeader.startsWith('Bearer ')) {
            res.status(401).json({
                success: false,
                error: 'Access token required'
            });
            return;
        }
        const token = authHeader.substring(7); // Remove 'Bearer ' prefix
        // Verify token with Supabase
        const { data: { user }, error } = await supabase_1.supabase.auth.getUser(token);
        if (error || !user) {
            res.status(401).json({
                success: false,
                error: 'Invalid or expired token'
            });
            return;
        }
        // Attach user to request
        req.user = user;
        next();
    }
    catch (error) {
        console.error('Authentication error:', error);
        res.status(500).json({
            success: false,
            error: 'Authentication failed'
        });
    }
};
exports.authenticateUser = authenticateUser;
/**
 * Middleware to require specific user role
 * Must be used after authenticateUser middleware
 */
const requireRole = (allowedRoles) => {
    return (req, res, next) => {
        if (!req.user) {
            res.status(401).json({
                success: false,
                error: 'Authentication required'
            });
            return;
        }
        // Check if user has required role
        const userRole = req.user.user_metadata?.role || 'vendor';
        if (!allowedRoles.includes(userRole)) {
            res.status(403).json({
                success: false,
                error: 'Insufficient permissions'
            });
            return;
        }
        next();
    };
};
exports.requireRole = requireRole;
/**
 * Middleware to require vendor role specifically
 */
exports.requireVendor = (0, exports.requireRole)(['vendor', 'admin', 'staff', 'superadmin']);
/**
 * Middleware to require admin role specifically
 */
exports.requireAdmin = (0, exports.requireRole)(['admin', 'superadmin']);
/**
 * Middleware to require super admin role specifically
 */
exports.requireSuperAdmin = (0, exports.requireRole)(['superadmin']);
//# sourceMappingURL=auth.js.map